#! /bin/sh

#
# runlex.sh
# Script to run Lex/Flex.
# First argument is the (quoted) name of the command; if it's null, that
# means that neither Flex nor Lex was found, so we report an error and
# quit.
#
# $Id$
#
# Wireshark - Network traffic analyzer
# By Gerald Combs <gerald@wireshark.org>
# Copyright 2007 Gerald Combs
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#

#
# Get the name of the command to run, and then shift to get the arguments.
#
if [ $# -eq 0 ]
then
	echo "Usage: runlex <lex/flex command to run> [ arguments ]" 1>&2
	exit 1
fi
LEX="$1"
shift
#
# Check whether we have it.
#
if [ -z "${LEX}" ]
then
	echo "Neither lex nor flex was found" 1>&2
	exit 1
fi

SED="$1"
shift
#
# Check whether we have it.
#
if [ -z "${SED}" ]
then
	echo "Sed was not found" 1>&2
	exit 1
fi

#
# Process the flags.  We don't use getopt because we don't want to
# embed complete knowledge of what options are supported by Lex/Flex.
#
flags=""
outfile=lex.yy.c
while [ $# -ne 0 ]
do
	case "$1" in

	-o*)
		#
		# Set the output file name.
		#
		outfile=`echo "$1" | ${SED} 's/-o\(.*\)/\1/'`
		;;

	-*)
		#
		# Add this to the list of flags.
		#
		flags="$flags $1"
		;;

	--|*)
		#
		# End of flags.
		#
		break
		;;
	esac
	shift
done

#
# OK, run it.
#
echo "Running ${LEX}"
${LEX} -o"$outfile" $flags "$@"

#
# Did it succeed?
#
if [ $? -ne 0 ]
then
	#
	# No.  Exit with the failing exit status.
	#
	exitstatus=$?
	echo "${LEX} failed: exit status $exitstatus"
	exit $exitstatus
fi

#
# Flex has the annoying habit of stripping all but the last component of
# the "-o" flag argument and using that as the place to put the output.
# This gets in the way of building in a directory different from the
# source directory.  Try to work around this.
#
# Is the outfile where we think it is?
#
outfile_base=`basename "$outfile"`
if [ "$outfile_base" != "$outfile" -a \( ! -r "$outfile" \) -a -r "$outfile_base" ]
then
	#
	# No, it's not, but it is in the current directory.  Put it
	# where it's supposed to be.
	#
	mv "$outfile_base" "$outfile"
	if [ $? -ne 0 ]
	then
		echo $?
	fi
fi

echo "Wrote $outfile"

#
# OK, now let's generate a header file declaring the relevant functions
# defined by the .c file; if the .c file is .../foo.c, the header file
# will be .../foo_lex.h.
#
# This works around some other Flex suckage, wherein it doesn't declare
# the lex routine before defining it, causing compiler warnings.
# XXX - newer versions of Flex support --header-file=, to generate the
# appropriate header file.  With those versions, we should use that option.
#

#
# Get the name of the prefix; scan the source files for a %option prefix
# line.  We use the last one.
#
echo "Getting prefix"
prefix=`${SED} -n 's/%option[ 	][ 	]*prefix="\(.*\)".*/\1/p' "$@" | tail -1`
if [ ! -z "$prefix" ]
then
	prefixline="#define yylex ${prefix}lex"
fi

#
# Construct the name of the header file.
#
echo "Getting header file name"
header_file=`dirname "$outfile"`/`basename "$outfile" .c`_lex.h

#
# Spew out the declaration.
#
echo "Writing $header_file"
cat <<EOF >$header_file
/* This is generated by runlex.sh.  Do not edit it. */
$prefixline
#ifndef YY_DECL
#define YY_DECL int yylex(void)
#endif
YY_DECL;
EOF

echo "Wrote $header_file"
